using System;
using System.Reflection;
using System.Globalization;
using System.Resources;
using System.Text;
using System.ComponentModel;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;

[AttributeUsage(AttributeTargets.All)]
internal sealed class SRDescriptionAttribute : DescriptionAttribute
{
    public SRDescriptionAttribute(string description)
    {
        DescriptionValue = SR.GetString(description);
    }

    public SRDescriptionAttribute(string description, string resourceSet)
    {
        ResourceManager rm = new ResourceManager(resourceSet, Assembly.GetExecutingAssembly());
        DescriptionValue = rm.GetString(description);
        System.Diagnostics.Debug.Assert(DescriptionValue != null, string.Format(CultureInfo.CurrentCulture, "String resource {0} not found.", new object[] { description }));
    }
}

[AttributeUsage(AttributeTargets.All)]
internal sealed class SRCategoryAttribute : CategoryAttribute
{
    private string resourceSet = String.Empty;

    public SRCategoryAttribute(string category)
        : base(category)
    {
    }

    public SRCategoryAttribute(string category, string resourceSet)
        : base(category)
    {
        this.resourceSet = resourceSet;
    }

    protected override string GetLocalizedString(string value)
    {
        if (this.resourceSet.Length > 0)
        {
            ResourceManager rm = new ResourceManager(resourceSet, Assembly.GetExecutingAssembly());
            String localizedString = rm.GetString(value);
            System.Diagnostics.Debug.Assert(localizedString != null, string.Format(CultureInfo.CurrentCulture, "String resource {0} not found.", new object[] { value }));
            return localizedString;
        }
        else
        {
            return SR.GetString(value);
        }
    }
}

[AttributeUsage(AttributeTargets.All)]
internal sealed class SRDisplayNameAttribute : DisplayNameAttribute
{
    public SRDisplayNameAttribute(string name)
    {
        DisplayNameValue = SR.GetString(name);
    }

    public SRDisplayNameAttribute(string name, string resourceSet)
    {
        ResourceManager rm = new ResourceManager(resourceSet, Assembly.GetExecutingAssembly());
        DisplayNameValue = rm.GetString(name);
        System.Diagnostics.Debug.Assert(DisplayNameValue != null, string.Format(CultureInfo.CurrentCulture, "String resource {0} not found.", new object[] { name }));
    }
}

/// <summary>
///    AutoGenerated resource class. Usage:
///
///        string s = SR.GetString(SR.MyIdenfitier);
/// </summary>
internal sealed class SR
{
    static SR loader = null;
    ResourceManager resources;

    internal SR()
    {
        resources = new System.Resources.ResourceManager("System.Workflow.ComponentModel.StringResources", Assembly.GetExecutingAssembly());
    }

    private static SR GetLoader()
    {
        if (loader == null)
            loader = new SR();
        return loader;
    }

    private static CultureInfo Culture
    {
        get { return null/*use ResourceManager default, CultureInfo.CurrentUICulture*/; }
    }

    [SuppressMessage("Microsoft.Performance", "CA1811:AvoidUncalledPrivateCode")]
    internal static string GetString(string name, params object[] args)
    {
        return GetString(SR.Culture, name, args);
    }

    [SuppressMessage("Microsoft.Performance", "CA1811:AvoidUncalledPrivateCode")]
    internal static string GetString(CultureInfo culture, string name, params object[] args)
    {
        SR sys = GetLoader();
        if (sys == null)
            return null;
        string res = sys.resources.GetString(name, culture);
        System.Diagnostics.Debug.Assert(res != null, string.Format(CultureInfo.CurrentCulture, "String resource {0} not found.", new object[] { name }));
        if (args != null && args.Length > 0)
        {
            return string.Format(CultureInfo.CurrentCulture, res, args);
        }
        else
        {
            return res;
        }
    }

    internal static string GetString(string name)
    {
        return GetString(SR.Culture, name);
    }

    internal static string GetString(CultureInfo culture, string name)
    {
        SR sys = GetLoader();
        if (sys == null)
            return null;
        string res = sys.resources.GetString(name, culture);
        System.Diagnostics.Debug.Assert(res != null, string.Format(CultureInfo.CurrentCulture, "String resource {0} not found.", new object[] { name }));
        return res;
    }

    // All these strings should be present in StringResources.resx
    internal const string Activity = "Activity";
    internal const string Handlers = "Handlers";
    internal const string Conditions = "Conditions";
    internal const string ConditionedActivityConditions = "ConditionedActivityConditions";
    internal const string Correlations = "Correlations";
    internal const string CorrelationSet = "CorrelationSet";
    internal const string NameDescr = "NameDescr";
    internal const string EnabledDescr = "EnabledDescr";
    internal const string DescriptionDescr = "DescriptionDescr";
    internal const string UnlessConditionDescr = "UnlessConditionDescr";
    internal const string InitializeDescr = "InitializeDescr";
    internal const string CatchTypeDescr = "CatchTypeDescr";
    internal const string ExceptionTypeDescr = "ExceptionTypeDescr";
    internal const string FaultDescription = "FaultDescription";
    internal const string FaultTypeDescription = "FaultTypeDescription";
    internal const string ContainingAssemblyDescr = "ContainingAssemblyDescr";
    internal const string ExecutionModeDescr = "ExecutionModeDescr";
    internal const string Error_ReadOnlyTemplateActivity = "Error_ReadOnlyTemplateActivity";
    internal const string Error_TypeNotString = "Error_TypeNotString";
    internal const string Error_InvalidErrorType = "Error_InvalidErrorType";
    internal const string Error_LiteralConversionFailed = "Error_LiteralConversionFailed";
    internal const string Error_TypeNotPrimitive = "Error_TypeNotPrimitive";
    internal const string CompletedCaleeDescr = "CompletedCaleeDescr";
    internal const string ProxyClassDescr = "ProxyClassDescr";
    internal const string ActivitySetDescr = "ActivitySetDescr";
    internal const string VersionDescr = "VersionDescr";
    internal const string ActivationDescr = "ActivationDescr";
    internal const string CorrelationSetsDescr = "CorrelationSetsDescr";
    internal const string CompanionClassDescr = "CompanionClassDescr";
    internal const string TransactionTypeDescr = "TransactionTypeDescr";
    internal const string SynchronizedDescr = "SynchronizedDescr";
    internal const string IsolationLevelDescr = "IsolationLevelDescr";
    internal const string TimeoutDescr = "TimeoutDescr";
    internal const string BatchableDescr = "BatchableDescr";
    internal const string LRTTimeoutDescr = "LRTTimeoutDescr";
    internal const string OnGetCalleeCountDescr = "OnGetCalleeCountDescr";
    internal const string CompensatableActivityDescr = "CompensatableActivityDescr";
    internal const string OnAfterEventDescr = "OnAfterEventDescr";
    internal const string OnBeforeMethodInvokeDescr = "OnBeforeMethodInvokeDescr";
    internal const string AssignedToDescr = "AssignedToDescr";
    internal const string TypeDescr = "TypeDescr";
    internal const string TemplateActivityDescr = "TemplateActivityDescr";
    internal const string ErrorMessageDescr = "ErrorMessageDescr";
    internal const string WebServiceSynchronizedDescr = "WebServiceSynchronizedDescr";
    internal const string CorrelationSetDescr = "CorrelationSetDescr";
    internal const string ExecutionTypeDescr = "ExecutionTypeDescr";
    internal const string RoleDescr = "RoleDescr";
    internal const string OnInitializeClonesDescr = "OnInitializeClonesDescr";
    internal const string CorrelationSetDisplayName = "CorrelationSetDisplayName";
    internal const string PastingActivities = "PastingActivities";
    internal const string DeletingActivities = "DeletingActivities";
    internal const string DragDropActivities = "DragDropActivities";
    internal const string ChangingEnabled = "ChangingEnabled";
    internal const string ChangingHandler = "ChangingHandler";
    internal const string ChangingParameter = "ChangingParameter";
    internal const string CollectionItem = "CollectionItem";
    internal const string AddingConditionalBranch = "AddingConditionalBranch";
    internal const string AddingEventActivity = "AddingEventActivity";
    internal const string AddingListenBranch = "AddingListenBranch";
    internal const string AddingParallelBranch = "AddingParallelBranch";
    internal const string CurrentProject = "CurrentProject";
    internal const string ReferencedAssemblies = "ReferencedAssemblies";
    internal const string CollectionText = "CollectionText";
    internal const string ParameterDescription = "ParameterDescription";
    internal const string InvokeParameterDescription = "InvokeParameterDescription";
    internal const string ParametersDescription = "ParametersDescription";
    internal const string ChangingParameters = "ChangingParameters";
    internal const string Condition = "ConditionRule";
    internal const string MovingActivities = "MovingActivities";
    internal const string MemberNameDescr = "MemberNameDescr";
    internal const string OnScopeInitializedDescr = "OnScopeInitializedDescr";
    internal const string OnGeneratorInitializedDescr = "OnGeneratorInitializedDescr";
    internal const string OnScopeCompletedDescr = "OnScopeCompletedDescr";
    internal const string OnGeneratorCompletedDescr = "OnGeneratorCompletedDescr";
    internal const string DataElementRuntimeTypeDescr = "DataElementRuntimeTypeDescr";
    internal const string RuleConditionReferencesDescr = "RuleConditionReferencesDescr";
    internal const string CreateActivityFromToolbox = "CreateActivityFromToolbox";
    internal const string MoveMultipleActivities = "MoveMultipleActivities";
    internal const string MoveSingleActivity = "MoveSingleActivity";
    internal const string CutMultipleActivities = "CutMultipleActivities";
    internal const string CutSingleActivity = "CutSingleActivity";
    internal const string CutActivity = "CutActivity";
    internal const string FaultActivityDescription = "FaultActivityDescription";
    internal const string NullConditionExpression = "NullConditionExpression";
    internal const string ParameterTypeDescription = "ParameterTypeDescription";
    internal const string ParameterCategory = "ParameterCategory";
    internal const string ParameterDirectionDescription = "ParameterDirectionDescription";
    internal const string ParameterElementDescription = "ParameterElementDescription";
    internal const string ParameterDlgDescription = "ParameterDlgDescription";
    internal const string ParameterDlgHeader = "ParameterDlgHeader";
    internal const string SuspendActivityDescription = "SuspendActivityDescription";
    internal const string SuspendErrorMessageDescr = "SuspendErrorMessageDescr";
    internal const string TerminateActivityDescription = "TerminateActivityDescription";
    internal const string TerminateErrorMessageDescr = "TerminateErrorMessageDescr";
    internal const string DeclarationCategory = "DeclarationCategory";
    internal const string NoValidActivityPropertiesAvailable = "NoValidActivityPropertiesAvailable";
    internal const string ChooseActivityDatasource = "ChooseActivityDatasource";
    internal const string Promote = "Promote";
    internal const string Type = "Type";
    internal const string NoMatchingActivityProperties = "NoMatchingActivityProperties";
    internal const string ActivityBindIDDescription = "ActivityBindIDDescription";
    internal const string ActivityBindPathDescription = "ActivityBindPathDescription";
    internal const string XPathDescription = "XPathDescription";
    internal const string TransformerDescription = "TransformerDescription";
    internal const string CustomPropertiesCollectionFormHeader = "CustomPropertiesCollectionFormHeader";
    internal const string CustomPropertiesCollectionFormDescription = "CustomPropertiesCollectionFormDescription";
    internal const string BaseTypePropertyName = "BaseTypePropertyName";
    internal const string CustomActivityBaseClassTypeFilterProviderDesc = "CustomActivityBaseClassTypeFilterProviderDesc";
    internal const string CustomActivityDesignerTypeFilterProviderDesc = "CustomActivityDesignerTypeFilterProviderDesc";
    internal const string CustomActivityValidatorTypeFilterProviderDesc = "CustomActivityValidatorTypeFilterProviderDesc";
    internal const string CustomActivityExecutorTypeFilterProviderDesc = "CustomActivityExecutorTypeFilterProviderDesc";
    internal const string GenericParameters = "GenericParameters";
    internal const string ToolboxItem = "ToolboxItem";
    internal const string ToolboxItemCompanionClassDesc = "ToolboxItemCompanionClassDesc";
    internal const string Error_SerializationInsufficientState = "Error_SerializationInsufficientState";
    internal const string Error_ActivityHasParent = "Error_ActivityHasParent";
    internal const string Error_CompensantionParentNotScope = "Error_CompensantionParentNotScope";
    internal const string Error_ConditionedActivityParentNotCAG = "Error_ConditionedActivityParentNotCAG";
    internal const string Error_CorrelationTypeNotComparable = "Error_CorrelationTypeNotComparable";
    internal const string Error_ArgumentTypeNotMatchParameter = "Error_ArgumentTypeNotMatchParameter";
    internal const string Error_TypeTypeMismatch = "Error_TypeTypeMismatch";
    internal const string Error_ParameterTypeMismatch = "Error_ParameterTypeMismatch";
    internal const string Error_InvokeParameterTypeMismatch = "Error_InvokeParameterTypeMismatch";
    internal const string Error_ParameterPropertyNotSet = "Error_ParameterPropertyNotSet";
    internal const string Error_DataSourceNameNotSet = "Error_DataSourceNameNotSet";
    internal const string Error_DataSourceInvalidIdentifier = "Error_DataSourceInvalidIdentifier";
    internal const string Error_ParameterTypeNotExist = "Error_ParameterTypeNotExist";
    internal const string Error_InvalidParameterName = "Error_InvalidParameterName";
    internal const string Error_InvalidParameterType = "Error_InvalidParameterType";
    internal const string Error_InvalidParameterElement = "Error_InvalidParameterElement";
    internal const string Error_InvalidPropertyType = "Error_InvalidPropertyType";
    internal const string Error_TypeNotResolvedInMethodName = "Error_TypeNotResolvedInMethodName";
    internal const string Error_DelegateNoInvoke = "Error_DelegateNoInvoke";
    internal const string Error_TypeNotDelegate = "Error_TypeNotDelegate";
    internal const string Error_MethodSignatureMismatch = "Error_MethodSignatureMismatch";
    internal const string Error_MethodReturnTypeMismatch = "Error_MethodReturnTypeMismatch";
    internal const string Error_PropertyNotSet = "Error_PropertyNotSet";
    internal const string Error_ScopeCouldNotBeResolved = "Error_ScopeCouldNotBeResolved";
    internal const string Error_IfElseNotAllIfElseBranchDecl = "Error_ConditionalNotAllConditionalBranchDecl";
    internal const string Error_TypeTypeMismatchAmbiguity = "Error_TypeTypeMismatchAmbiguity";
    internal const string Error_InvalidCorrelationSetDatasource = "Error_InvalidCorrelationSetDatasource";
    internal const string Error_InvalidCorrelationSetType = "Error_InvalidCorrelationSetType";
    internal const string Error_MissingCorrelationParameterAttribute = "Error_MissingCorrelationParameterAttribute";
    internal const string Error_CorrelationTypeNotConsistent = "Error_CorrelationTypeNotConsistent";
    internal const string Error_CorrelationInvalid = "Error_CorrelationInvalid";
    internal const string Error_MissingDelegateMethod = "Error_MissingDelegateMethod";
    internal const string Error_MissingHostInterface = "Error_MissingHostInterface";
    internal const string Error_MissingMethodName = "Error_MissingMethodName";
    internal const string Error_NoBoundType = "Error_NoBoundType";
    internal const string Error_PortTypeNotAnInterface = "Error_PortTypeNotAnInterface";
    internal const string Error_MethodNotExists = "Error_MethodNotExists";
    internal const string Error_InvalidRequestResponseMethod = "Error_InvalidRequestResponseMethod";
    internal const string General_MissingService = "General_MissingService";
    internal const string Error_ScopeDuplicatedNameActivity = "Error_ScopeDuplicatedNameActivity";
    internal const string Error_DuplicatedActivityID = "Error_DuplicatedActivityID";
    internal const string Error_DuplicatedParameterName = "Error_DuplicatedParameterName";
    internal const string Error_ScopeMissingSerializableAttribute = "Error_ScopeMissingSerializableAttribute";
    internal const string Error_FieldNotExists = "Error_FieldNotExists";
    internal const string Error_PropertyNotExists = "Error_PropertyNotExists";
    internal const string Error_FieldTypeMismatch = "Error_FieldTypeMismatch";
    internal const string Error_PropertyTypeMismatch = "Error_PropertyTypeMismatch";
    internal const string Error_TypeNotResolvedInFieldName = "Error_TypeNotResolvedInFieldName";
    internal const string Error_TypeNotResolvedInPropertyName = "Error_TypeNotResolvedInPropertyName";
    internal const string Error_FieldGenericParamTypeMismatch = "Error_FieldGenericParamTypeMismatch";
    internal const string Error_TypeNotResolved = "Error_TypeNotResolved";
    internal const string Error_TypeIsUnboundedGeneric = "Error_TypeIsUnboundedGeneric";
    internal const string Error_MissingRootActivity = "Error_MissingRootActivity";
    internal const string Error_PropertyNotReadable = "Error_PropertyNotReadable";
    internal const string Error_PropertyNotWritable = "Error_PropertyNotWritable";
    internal const string Error_NotCompositeActivity = "Error_NotCompositeActivity";
    internal const string Error_TypeNotExist = "Error_TypeNotExist";
    internal const string Error_ActivityRefNotResolved = "Error_ActivityRefNotResolved";
    internal const string Error_ActivityRefNotMatchType = "Error_ActivityRefNotMatchType";
    internal const string Error_ActivityValidation = "Error_ActivityValidation";
    internal const string Error_ActiveChildExist = "Error_ActiveChildExist";
    internal const string Error_ActiveChildContextExist = "Error_ActiveChildContextExist";
    internal const string Error_CannotCompleteContext = "Error_CannotCompleteContext";
    internal const string Error_NoPasteSupport = "Error_NoPasteSupport";
    internal const string Error_UnknownSerializationStore = "Error_UnknownSerializationStore";
    internal const string Error_MissingCorrelationSet = "Error_MissingCorrelationSet";
    internal const string Error_CreateVariable = "Error_CreateVariable";
    internal const string Error_DuplicateCorrelationSetName = "Error_DuplicateCorrelationSetName";
    internal const string Error_DragDropInvalid = "Error_DragDropInvalid";
    internal const string AddingImplicitActivity = "AddingImplicitActivity";
    internal const string Failure_DoDefaultAction = "Failure_DoDefaultAction";
    internal const string Failure_DoDefaultActionCaption = "Failure_DoDefaultActionCaption";
    internal const string Error_FaultInsideAtomicScope = "Error_FaultInsideAtomicScope";
    internal const string Error_ListenNotMoreThanOneDelay = "Error_ListenNotMoreThanOneDelay";
    internal const string Error_AtomicScopeWithFaultHandlersActivityDecl = "Error_AtomicScopeWithFaultHandlersActivityDecl";
    internal const string Error_AtomicScopeWithCancellationHandlerActivity = "Error_AtomicScopeWithCancellationHandlerActivity";
    internal const string Error_ScopeDuplicateFaultHandlerActivityForAll = "Error_ScopeDuplicateFaultHandlerActivityForAll";
    internal const string Error_ScopeDuplicateFaultHandlerActivityFor = "Error_ScopeDuplicateFaultHandlerActivityFor";
    internal const string Error_AtomicScopeNestedInNonLRT = "Error_AtomicScopeNestedInNonLRT";
    internal const string Error_LRTScopeNestedInNonLRT = "Error_LRTScopeNestedInNonLRT";
    internal const string Error_CAGNotAllChildrenConditioned = "Error_CAGNotAllChildrenConditioned";
    internal const string Error_ConditionedActivityChildCount = "Error_ConditionedActivityChildCount";
    internal const string Error_NegativeValue = "Error_NegativeValue";
    internal const string Error_MethodWithReturnType = "Error_MethodWithReturnType";
    internal const string Error_SendReceiveOrderIncorrect = "Error_SendReceiveOrderIncorrect";
    internal const string Error_ReceiveSendOrderIncorrect = "Error_ReceiveSendOrderIncorrect";
    internal const string Error_CompensateBadNesting = "Error_CompensateBadNesting";
    internal const string Error_ReferencedAssemblyIsInvalid = "Error_ReferencedAssemblyIsInvalid";
    internal const string Error_TypeToXsdConversion = "Error_TypeToXsdConversion";
    internal const string Error_FieldTypeNotResolved = "Error_FieldTypeNotResolved";
    internal const string Error_PropertyTypeNotResolved = "Error_PropertyTypeNotResolved";
    internal const string Error_CouldNotDeserializeXomlFile = "Error_CouldNotDeserializeXomlFile";
    internal const string Error_InternalCompilerError = "Error_InternalCompilerError";
    internal const string Error_TypeNotAsseblyQualified = "Error_TypeNotAsseblyQualified";
    internal const string CompilerWarning_StandardAssemlbyInReferences = "CompilerWarning_StandardAssemlbyInReferences";
    internal const string Error_SuspendInAtomicScope = "Error_SuspendInAtomicScope";
    internal const string Error_InvalidActivityExecutionContext = "Error_InvalidActivityExecutionContext";
    internal const string Error_NoRuntimeAvailable = "Error_NoRuntimeAvailable";
    internal const string Error_CanNotChangeAtRuntime = "Error_CanNotChangeAtRuntime";
    internal const string Error_DataContextNotInitialized = "Error_DataContextNotInitialized";
    internal const string Error_DataContextAlreadyInitialized = "Error_DataContextAlreadyInitialized";
    internal const string Error_ParseActivityNameDoesNotExist = "Error_ParseActivityNameDoesNotExist";
    internal const string Error_NoParameterPropertyDeclared = "Error_NoParameterPropertyDeclared";
    internal const string Error_PropertyInvalidIdentifier = "Error_PropertyInvalidIdentifier";
    internal const string Error_WorkflowDefinitionModified = "Error_WorkflowDefinitionModified";
    internal const string Error_FieldAlreadyExist = "Error_FieldAlreadyExist";
    internal const string Failure_FieldAlreadyExist = "Failure_FieldAlreadyExist";
    internal const string Error_DifferentTypeFieldExists = "Error_DifferentTypeFieldExists";
    internal const string Error_RootActivityTypeInvalid = "Error_RootActivityTypeInvalid";
    internal const string Error_RootActivityTypeInvalid2 = "Error_RootActivityTypeInvalid2";
    internal const string Error_CannotCompile_No_XClass = "Error_CannotCompile_No_XClass";
    internal const string Error_TemplateActivityIsNotActivity = "Error_TemplateActivityIsNotActivity";
    internal const string Error_TypeIsNotRootActivity = "Error_TypeIsNotRootActivity";
    internal const string Error_NoTypeProvider = "Error_NoTypeProvider";
    internal const string Error_NotCodeGeneratorType = "Error_NotCodeGeneratorType";
    internal const string Error_NotDataContext = "Error_NotDataContext";
    internal const string Error_MissingDefaultConstructor = "Error_MissingDefaultConstructor";
    internal const string Error_ContextStackItemMissing = "Error_ContextStackItemMissing";
    internal const string Error_UnexpectedArgumentType = "Error_UnexpectedArgumentType";
    internal const string Error_EmptyArgument = "Error_EmptyArgument";
    internal const string Error_DPAlreadyExist = "Error_DPAlreadyExist";
    internal const string Error_DuplicateDynamicProperty = "Error_DuplicateDynamicProperty";
    internal const string Error_DynamicPropertyTypeValueMismatch = "Error_DynamicPropertyTypeValueMismatch";
    internal const string Error_DynamicPropertyNoSupport = "Error_DynamicPropertyNoSupport";
    internal const string Error_NoContextForDatasource = "Error_NoContextForDatasource";
    internal const string Error_NoContextForDatasourceCaption = "Error_NoContextForDatasourceCaption";
    internal const string Error_DataSourceHasParent = "Error_DataSourceHasParent";
    internal const string OnTaskCompletedDescr = "OnTaskCompletedDescr";
    internal const string OnTaskInitializedDescr = "OnTaskInitializedDescr";
    internal const string Error_InvalidXmlData = "Error_InvalidXmlData";
    internal const string Error_HandlerNotOnRoot = "Error_HandlerNotOnRoot";
    internal const string Error_InvalidArgumentIndex = "Error_InvalidArgumentIndex";
    internal const string Error_UITypeEditorTypeNotUITypeEditor = "Error_UITypeEditorTypeNotUITypeEditor";
    internal const string FilterDescription_UITypeEditor = "FilterDescription_UITypeEditor";
    internal const string Error_UserCodeFilesNotAllowed = "Error_UserCodeFilesNotAllowed";
    internal const string Error_CodeWithinNotAllowed = "Error_CodeWithinNotAllowed";
    internal const string Error_TypeNotAuthorized = "Error_TypeNotAuthorized";
    internal const string Error_CantDetermineBaseType = "Error_CantDetermineBaseType";
    internal const string Error_MultipleSelectNotSupportedForBindAndPromote = "Error_MultipleSelectNotSupportedForBindAndPromote";
    internal const string Error_CantDetermineBaseTypeCaption = "Error_CantDetermineBaseTypeCaption";
    internal const string Error_CantDeterminePropertyBaseType = "Error_CantDeterminePropertyBaseType";
    internal const string Error_NullCustomActivityTypeName = "Error_NullCustomActivityTypeName";
    internal const string Error_InvalidAttribute = "Error_InvalidAttribute";
    internal const string Error_InvalidAttributes = "Error_InvalidAttributes";
    internal const string Error_ConfigFileMissingOrInvalid = "Error_ConfigFileMissingOrInvalid";
    internal const string Error_CantHaveContextActivity = "Error_CantHaveContextActivity";
    internal const string Error_SynchronizedNeedsDataContext = "Error_SynchronizedNeedsDataContext";
    internal const string Error_MoreThanOneFaultHandlersActivityDecl = "Error_MoreThanOneFaultHandlersActivityDecl";
    internal const string Error_MoreThanOneEventHandlersDecl = "Error_MoreThanOneEventHandlersDecl";
    internal const string Error_MoreThanOneCancelHandler = "Error_MoreThanOneCancelHandler";
    internal const string Error_MetaDataInterfaceMissing = "Error_MetaDataInterfaceMissing";
    internal const string Error_NonActivityExecutor = "Error_NonActivityExecutor";
    internal const string Error_DynamicUpdateEvaluation = "Error_DynamicUpdateEvaluation";
    internal const string Error_CollectionHasNullEntry = "Error_CollectionHasNullEntry";
    internal const string Error_MissingContextProperty = "Error_MissingContextProperty";
    internal const string Error_AssociatedDesignerMissing = "Error_AssociatedDesignerMissing";
    internal const string Error_MissingContextActivityProperty = "Error_MissingContextActivityProperty";
    internal const string Error_MissingActivityProperty = "Error_MissingActivityProperty";
    internal const string Error_MissingOwnerTypeProperty = "Error_MissingOwnerTypeProperty";
    internal const string Error_DOIsNotAnActivity = "Error_DOIsNotAnActivity";
    internal const string Error_PropertyCanBeOnlyCleared = "Error_PropertyCanBeOnlyCleared";
    internal const string Error_PropertyDefaultTypeMismatch = "Error_PropertyDefaultTypeMismatch";
    internal const string Error_PropertyDefaultIsReference = "Error_PropertyDefaultIsReference";
    // workflow load errors
    internal const string Error_WorkflowLoadFailed = "Error_WorkflowLoadFailed";
    internal const string Error_WorkflowLoadValidationFailed = "Error_WorkflowLoadValidationFailed";
    internal const string Error_WorkflowLoadDeserializationFailed = "Error_WorkflowLoadDeserializationFailed";
    internal const string Error_WorkflowLoadTypeMismatch = "Error_WorkflowLoadTypeMismatch";
    internal const string Error_WorkflowLoadInvalidXoml = "Error_WorkflowLoadInvalidXoml";
    internal const string Error_WorkflowLoadNotValidRootType = "Error_WorkflowLoadNotValidRootType";
    internal const string Error_CantCreateInstanceOfComponent = "Error_CantCreateInstanceOfComponent";
    internal const string Error_NotComponentFactoryType = "Error_NotComponentFactoryType";
    internal const string Error_WorkflowTerminated = "Error_WorkflowTerminated";

    // serializer errrors
    internal const string Error_SerializerAttributesFoundInComplexProperty = "Error_SerializerAttributesFoundInComplexProperty";
    internal const string Error_InvalidDataFound = "Error_InvalidDataFound";
    internal const string Error_InvalidDataFoundForType = "Error_InvalidDataFoundForType";
    internal const string Error_InvalidDataFoundForType1 = "Error_InvalidDataFoundForType1";
    internal const string Error_SerializerTypeNotResolved = "Error_SerializerTypeNotResolved";
    internal const string Error_MarkupSerializerTypeNotResolved = "Error_MarkupSerializerTypeNotResolved";
    internal const string Error_SerializerTypeNotResolvedWithInnerError = "Error_SerializerTypeNotResolvedWithInnerError";
    internal const string Error_SerializerNotAvailable = "Error_SerializerNotAvailable";
    internal const string Error_SerializerNotAvailableForSerialize = "Error_SerializerNotAvailableForSerialize";
    internal const string Error_SerializerCreateInstanceFailed = "Error_SerializerCreateInstanceFailed";
    internal const string Error_SerializerAddChildFailed = "Error_SerializerAddChildFailed";
    internal const string Error_SerializerNoPropertyAvailable = "Error_SerializerNoPropertyAvailable";
    internal const string Error_SerializerPrimitivePropertyReadOnly = "Error_SerializerPrimitivePropertyReadOnly";
    internal const string Error_SerializerCantChangeIsLocked = "Error_SerializerCantChangeIsLocked";
    internal const string Error_SerializerPrimitivePropertySetFailed = "Error_SerializerPrimitivePropertySetFailed";
    internal const string Error_SerializerPropertyGetFailed = "Error_SerializerPropertyGetFailed";
    internal const string Error_SerializerPrimitivePropertyNoLogic = "Error_SerializerPrimitivePropertyNoLogic";
    internal const string Error_SerializerPrimitivePropertyParentIsNull = "Error_SerializerPrimitivePropertyParentIsNull";
    internal const string Error_SerializerComplexPropertySetFailed = "Error_SerializerComplexPropertySetFailed";
    internal const string Error_SerializerNoChildNotion = "Error_SerializerNoChildNotion";
    internal const string Error_SerializerNoDynamicPropertySupport = "Error_SerializerNoDynamicPropertySupport";
    internal const string Error_SerializerNoSerializeLogic = "Error_SerializerNoSerializeLogic";
    internal const string Error_SerializerReadOnlyPropertyAndValueIsNull = "Error_SerializerReadOnlyPropertyAndValueIsNull";
    internal const string Error_SerializerReadOnlyParametersNoChild = "Error_SerializerReadOnlyParametersNoChild";
    internal const string Error_SerializerNotParameterBindingObject = "Error_SerializerNotParameterBindingObject";
    internal const string Error_SerializerThrewException = "Error_SerializerThrewException";
    internal const string Error_ActivityCollectionSerializer = "Error_ActivityCollectionSerializer";
    internal const string Error_MissingClassAttribute = "Error_MissingClassAttribute";
    internal const string Error_MissingClassAttributeValue = "Error_MissingClassAttributeValue";
    internal const string ExecutorCreationFailedErrorMessage = "ExecutorCreationFailedErrorMessage";
    internal const string VariableGetterCode_VB = "VariableGetterCode_VB";
    internal const string VariableGetterCode_CS = "VariableGetterCode_CS";
    internal const string VariableSetterCode_VB = "VariableSetterCode_VB";
    internal const string VariableSetterCode_CS = "VariableSetterCode_CS";
    internal const string StaticVariableGetterCode_VB = "StaticVariableGetterCode_VB";
    internal const string StaticVariableGetterCode_CS = "StaticVariableGetterCode_CS";
    internal const string StaticVariableSetterCode_VB = "StaticVariableSetterCode_VB";
    internal const string StaticVariableSetterCode_CS = "StaticVariableSetterCode_CS";
    internal const string EnterCodeBesidesCode_VB = "EnterCodeBesidesCode_VB";
    internal const string EnterCodeBesidesCode_CS = "EnterCodeBesidesCode_CS";
    internal const string LeaveCodeBesides1Code_VB = "LeaveCodeBesides1Code_VB";
    internal const string LeaveCodeBesides2Code_VB = "LeaveCodeBesides2Code_VB";
    internal const string LeaveCodeBesides1Code_CS = "LeaveCodeBesides1Code_CS";
    internal const string LeaveCodeBesides2Code_CS = "LeaveCodeBesides2Code_CS";
    internal const string VariableSetterName = "VariableSetterName";
    internal const string VariableGetterName = "VariableGetterName";
    internal const string HandlerGetterName = "HandlerGetterName";
    internal const string WorkflowCreatorName = "WorkflowCreatorName";
    internal const string ActivityMethod = "ActivityMethod";
    internal const string CustomActivityPrivateField = "CustomActivityPrivateField";
    internal const string InitializedVariableDeclaration_VB = "InitializedVariableDeclaration_VB";
    internal const string InitializedVariableDeclaration_CS = "InitializedVariableDeclaration_CS";
    internal const string In = "In";
    internal const string Out = "Out";
    internal const string Ref = "Ref";
    internal const string Required = "Required";
    internal const string Optional = "Optional";
    internal const string Parameters = "Parameters";
    internal const string Properties = "Properties";
    internal const string Error_RecursionDetected = "Error_RecursionDetected";
    internal const string Warning_UnverifiedRecursion = "Warning_UnverifiedRecursion";
    internal const string AddConstructorCode = "AddConstructorCode";
    internal const string Error_UninitializedCorrelation = "Error_UninitializedCorrelation";
    internal const string Error_CorrelationAlreadyInitialized = "Error_CorrelationAlreadyInitialized";
    internal const string Error_CorrelatedSendReceiveAtomicScope = "Error_CorrelatedSendReceiveAtomicScope";
    internal const string Warning_ActivityValidation = "Warning_ActivityValidation";
    internal const string Warning_EmptyBehaviourActivity = "Warning_EmptyBehaviourActivity";
    internal const string Error_ParallelActivationNoCorrelation = "Error_ParallelActivationNoCorrelation";
    internal const string Error_MethodNotAccessible = "Error_MethodNotAccessible";
    internal const string Error_FieldNotAccessible = "Error_FieldNotAccessible";
    internal const string Error_PropertyNotAccessible = "Error_PropertyNotAccessible";
    internal const string Error_GenericArgumentsNotAllowed = "Error_GenericArgumentsNotAllowed";
    internal const string Error_InvalidIdentifier = "Error_InvalidIdentifier";
    internal const string Error_InvalidLanguageIdentifier = "Error_InvalidLanguageIdentifier";
    internal const string DuplicateActivityIdentifier = "DuplicateActivityIdentifier";
    internal const string Error_MissingAttribute = "Error_MissingAttribute";
    internal const string Error_LoadUIPropertiesFile = "Error_LoadUIPropertiesFile";
    internal const string Error_SerializerEventGetFailed = "Error_SerializerEventGetFailed";
    internal const string Error_SerializerEventFailed = "Error_SerializerEventFailed";
    internal const string Error_SerializerNoMemberFound = "Error_SerializerNoMemberFound";
    internal const string Error_DynamicEventConflict = "Error_DynamicEventConflict";
    internal const string Error_SerializerMemberSetFailed = "Error_SerializerMemberSetFailed";
    internal const string Error_ContentPropertyCouldNotBeFound = "Error_ContentPropertyCouldNotBeFound";
    internal const string Error_ContentPropertyValueInvalid = "Error_ContentPropertyValueInvalid";
    internal const string Error_ContentPropertyNoSetter = "Error_ContentPropertyNoSetter";
    internal const string Error_ContentCanNotBeConverted = "Error_ContentCanNotBeConverted";
    internal const string Error_ContentPropertyCanNotBeNull = "Error_ContentPropertyCanNotBeNull";
    internal const string Error_SerializerTypeMismatch = "Error_SerializerTypeMismatch";
    internal const string Error_CouldNotAddValueInContentProperty = "Error_CouldNotAddValueInContentProperty";
    internal const string Error_SerializerTypeRequirement = "Error_SerializerTypeRequirement";
    internal const string Error_CanNotAddActivityInBlackBoxActivity = "Error_CanNotAddActivityInBlackBoxActivity";
    internal const string Error_ContentPropertyCanNotSupportCompactFormat = "Error_ContentPropertyCanNotSupportCompactFormat";
    internal const string Error_ContentPropertyNoMultipleContents = "Error_ContentPropertyNoMultipleContents";
    internal const string Error_InternalSerializerError = "Error_InternalSerializerError";
    internal const string Error_DictionarySerializerNonDictionaryObject = "Error_DictionarySerializerNonDictionaryObject";
    internal const string Error_DictionarySerializerKeyNotFound = "Error_DictionarySerializerKeyNotFound";
    internal const string Error_InvalidCancelActivityState = "Error_InvalidCancelActivityState";
    internal const string Error_InvalidCompensateActivityState = "Error_InvalidCompensateActivityState";
    internal const string Error_InvalidCloseActivityState = "Error_InvalidCloseActivityState";
    internal const string Error_SealedPropertyMetadata = "Error_SealedPropertyMetadata";
    internal const string Error_MemberNotFound = "Error_MemberNotFound";
    internal const string Error_EmptyPathValue = "Error_EmptyPathValue";
    internal const string Error_InvalidCompensatingState = "Error_InvalidCompensatingState";
    internal const string Error_InvalidCancelingState = "Error_InvalidCancelingState";
    internal const string Error_InvalidClosingState = "Error_InvalidClosingState";
    internal const string Error_InvalidStateToExecuteChild = "Error_InvalidStateToExecuteChild";
    internal const string Error_InvalidExecutionState = "Error_InvalidExecutionState";
    internal const string Error_InvalidInitializingState = "Error_InvalidInitializingState";
    internal const string Error_InvalidInvokingState = "Error_InvalidInvokingState";
    internal const string Error_NotRegisteredAs = "Error_NotRegisteredAs";
    internal const string Error_AlreadyRegisteredAs = "Error_AlreadyRegisteredAs";
    internal const string Error_InsertingChildControls = "Error_InsertingChildControls";
    internal const string Error_EmptyToolTipRectangle = "Error_EmptyToolTipRectangle";
    internal const string Error_EmptyRectangleValue = "Error_EmptyRectangleValue";
    internal const string Error_InvalidShadowRectangle = "Error_InvalidShadowRectangle";
    internal const string Error_InvalidShadowDepth = "Error_InvalidShadowDepth";
    internal const string Error_InvalidLightSource = "Error_InvalidLightSource";
    internal const string Error_ChangingDock = "Error_ChangingDock";
    internal const string Error_NullOrEmptyValue = "Error_NullOrEmptyValue";
    internal const string Error_InvalidStateImages = "Error_InvalidStateImages";
    internal const string Error_InvalidConnectorSegment = "Error_InvalidConnectorSegment";
    internal const string Error_InvalidConnectorSource = "Error_InvalidConnectorSource";
    internal const string Error_CreatingToolTip = "Error_CreatingToolTip";
    internal const string Error_InvalidDockStyle = "Error_InvalidDockStyle";
    internal const string Error_InvalidConnectorValue = "Error_InvalidConnectorValue";
    internal const string Error_InvalidDesignerVerbValue = "Error_InvalidDesignerVerbValue";
    internal const string Error_InvalidRuntimeType = "Error_InvalidRuntimeType";
    internal const string Error_InvalidArgumentValue = "Error_InvalidArgumentValue";
    internal const string Error_InvalidRadiusValue = "Error_InvalidRadiusValue";
    internal const string ToolTipString = "ToolTipString";

    //Collection Editor Resources
    internal const string CollectionEditorCaption = "CollectionEditorCaption";
    internal const string CollectionEditorProperties = "CollectionEditorProperties";
    internal const string CollectionEditorPropertiesMultiSelect = "CollectionEditorPropertiesMultiSelect";
    internal const string CollectionEditorPropertiesNone = "CollectionEditorPropertiesNone";
    internal const string CollectionEditorCantRemoveItem = "CollectionEditorCantRemoveItem";
    internal const string CollectionEditorUndoBatchDesc = "CollectionEditorUndoBatchDesc";
    internal const string CollectionEditorInheritedReadOnlySelection = "CollectionEditorInheritedReadOnlySelection";
    internal const string Error_ParameterAlreadyExists = "Error_ParameterAlreadyExists";
    internal const string Error_PropertyAlreadyExists = "Error_PropertyAlreadyExists";
    internal const string Error_HiddenPropertyAlreadyExists = "Error_HiddenPropertyAlreadyExists";
    internal const string Error_CorrelationInUse = "Error_CorrelationInUse";
    internal const string Error_ItemNotExists = "Error_ItemNotExists";
    internal const string Error_NoHelpAvailable = "Error_NoHelpAvailable";
    internal const string Error_DuplicateWorkflow = "Error_DuplicateWorkflow";
    internal const string Error_Recursion = "Error_Recursion";
    internal const string Error_RootActivity = "Error_RootActivity";
    internal const string Error_ConditionDefinitionDeserializationFailed = "Error_ConditionDefinitionDeserializationFailed";
    internal const string Error_InvalidConditionDefinition = "Error_InvalidConditionDefinition";
    internal const string SR_InvokeTransactionalFromAtomic = "SR_InvokeTransactionalFromAtomic";
    internal const string Error_SuspendInAtomicCallChain = "Error_SuspendInAtomicCallChain";
    internal const string Error_LiteralPassedToOutRef = "Error_LiteralPassedToOutRef";
    internal const string Error_GeneratorShouldContainSingleActivity = "Error_GeneratorShouldContainSingleActivity";
    internal const string Error_DeclaringPropertyNotSupported = "Error_DeclaringPropertyNotSupported";
    internal const string Error_DeclaringEventNotSupported = "Error_DeclaringEventNotSupported";
    internal const string Error_DynamicEventNotSupported = "Error_DynamicEventNotSupported";
    internal const string Error_DynamicPropertyNotSupported = "Error_DynamicPropertyNotSupported";
    internal const string Error_ParameterTypeResolution = "Error_ParameterTypeResolution";

    // Dynamic Validations
    internal const string Error_DynamicActivity = "Error_DynamicActivity";
    internal const string Error_DynamicActivity2 = "Error_DynamicActivity2";
    internal const string Error_CompilerValidationFailed = "Error_CompilerValidationFailed";
    internal const string Error_RuntimeValidationFailed = "Error_RuntimeValidationFailed";
    internal const string Error_TransactionAlreadyCanceled = "Error_TransactionAlreadyCanceled";
    internal const string Error_RemoveExecutingActivity = "Error_RemoveExecutingActivity";
    internal const string Error_InsideAtomicScope = "Error_InsideAtomicScope";
    internal const string SuspendReason_WorkflowChange = "SuspendReason_WorkflowChange";

    //type filtering
    internal const string FilterDescription_ParameterDeclaration = "FilterDescription_ParameterDeclaration";
    internal const string FilterDescription_GenericArgument = "FilterDescription_GenericArgument";


    internal const string LibraryPathIsInvalid = "LibraryPathIsInvalid";

    // Activity Set
    internal const string Error_CreateValidator = "Error_CreateValidator";
    internal const string Error_InvalidPackageFile = "Error_InvalidPackageFile";
    internal const string Error_AddAssemblyRef = "Error_AddAssemblyRef";
    internal const string Error_AssemblyBadImage = "Error_AssemblyBadImage";
    internal const string BindPropertySetterName = "BindPropertySetterName";

    // Bind validations
    internal const string Error_CannotResolveActivity = "Error_CannotResolveActivity";
    internal const string Error_CannotResolveRelativeActivity = "Error_CannotResolveRelativeActivity";
    internal const string Error_PathNotSetForActivitySource = "Error_PathNotSetForActivitySource";
    internal const string Error_InvalidMemberPath = "Error_InvalidMemberPath";
    internal const string Error_TargetTypeMismatch = "Error_TargetTypeMismatch";
    internal const string Warning_ParameterBinding = "Warning_ParameterBinding";
    internal const string Error_ReferencedActivityPropertyNotBind = "Error_ReferencedActivityPropertyNotBind";
    internal const string Error_TargetTypeDataSourcePathMismatch = "Error_TargetTypeDataSourcePathMismatch";
    internal const string Bind_ActivityDataSourceRecursionDetected = "Bind_ActivityDataSourceRecursionDetected";
    internal const string Bind_DuplicateDataSourceNames = "Bind_DuplicateDataSourceNames";
    internal const string Error_PathNotSetForXmlDataSource = "Error_PathNotSetForXmlDataSource";
    internal const string Error_XmlDocumentLoadFailed = "Error_XmlDocumentLoadFailed";
    internal const string Error_XmlDataSourceInvalidPath = "Error_XmlDataSourceInvalidPath";
    internal const string Error_XmlDataSourceMultipleNodes = "Error_XmlDataSourceMultipleNodes";
    internal const string Error_XmlDataSourceInvalidXPath = "Error_XmlDataSourceInvalidXPath";
    internal const string Error_InvalidObjectRefFormat = "Error_InvalidObjectRefFormat";
    internal const string Error_ReadOnlyDataSource = "Error_ReadOnlyDataSource";
    internal const string Error_HandlerReadOnly = "Error_HandlerReadOnly";
    internal const string Error_XmlDataSourceReadOnly = "Error_XmlDataSourceReadOnly";
    internal const string Error_DataSourceNotExist = "Error_DataSourceNotExist";
    internal const string Error_PropertyNoGetter = "Error_PropertyNoGetter";
    internal const string Error_PropertyNoSetter = "Error_PropertyNoSetter";
    internal const string Error_PropertyHasNoGetterDefined = "Error_PropertyHasNoGetterDefined";
    internal const string Error_PropertyHasNoSetterDefined = "Error_PropertyHasNoSetterDefined";
    internal const string Error_PropertyReferenceNoGetter = "Error_PropertyReferenceNoGetter";
    internal const string Error_PropertyReferenceGetterNoAccess = "Error_PropertyReferenceGetterNoAccess";
    internal const string Error_PropertyHasIndexParameters = "Error_PropertyHasIndexParameters";
    internal const string Error_ReadOnlyField = "Error_ReadOnlyField";
    internal const string Error_NoEnclosingContext = "Error_NoEnclosingContext";
    internal const string Error_NestedPersistOnClose = "Error_NestedPersistOnClose";
    internal const string Error_NestedCompensatableActivity = "Error_NestedCompensatableActivity";
    internal const string Error_InvalidActivityForObjectDatasource = "Error_InvalidActivityForObjectDatasource";
    internal const string Error_DataSourceTypeConversionFailed = "Error_DataSourceTypeConversionFailed";
    internal const string Error_BindDialogWrongPropertyType = "Error_BindDialogWrongPropertyType";
    internal const string Error_BindDialogNoValidPropertySelected = "Error_BindDialogNoValidPropertySelected";
    internal const string Error_BindDialogBindNotValid = "Error_BindDialogBindNotValid";
    internal const string Error_BindDialogCanNotBindToItself = "Error_BindDialogCanNotBindToItself";
    internal const string Error_BindActivityReference = "Error_BindActivityReference";
    internal const string Error_NoTargetTypeForMethod = "Error_NoTargetTypeForMethod";
    internal const string Error_MethodDataSourceIsReadOnly = "Error_MethodDataSourceIsReadOnly";
    internal const string Error_NotMethodDataSource = "Error_NotMethodDataSource";
    internal const string Error_MethodDataSourceWithPath = "Error_MethodDataSourceWithPath";
    internal const string Error_PathSyntax = "Error_PathSyntax";
    internal const string Error_UnmatchedParen = "Error_UnmatchedParen";
    internal const string Error_UnmatchedBracket = "Error_UnmatchedBracket";
    internal const string Error_MemberWithSameNameExists = "Error_MemberWithSameNameExists";
    internal const string Error_ActivityIdentifierCanNotBeEmpty = "Error_ActivityIdentifierCanNotBeEmpty";
    internal const string Error_InvalidActivityIdentifier = "Error_InvalidActivityIdentifier";
    internal const string Error_ActivityBindTypeConversionError = "Error_ActivityBindTypeConversionError";
    internal const string EmptyValue = "EmptyValue";
    internal const string Error_PropertyTypeNotDefined = "Error_PropertyTypeNotDefined";

    internal const string Error_CompilationFailed = "Error_CompilationFailed";
    internal const string Error_MissingCompilationContext = "Error_MissingCompilationContext";

    internal const string InvokeWorkflowReference_VB = "InvokeWorkflowReference_VB";
    internal const string InvokeWorkflowReference_CS = "InvokeWorkflowReference_CS";
    internal const string Error_InvalidListItem = "Error_InvalidListItem";

    internal const string ParserMapPINoWhitespace = "ParserMapPINoWhitespace";
    internal const string ParserMapPIBadCharEqual = "ParserMapPIBadCharEqual";
    internal const string ParserMapPIBadCharQuote = "ParserMapPIBadCharQuote";
    internal const string ParserMapPIBadKey = "ParserMapPIBadKey";
    internal const string ParserMapPIMissingKey = "ParserMapPIMissingKey";
    internal const string ParserMapPIKeyNotSet = "ParserMapPIKeyNotSet";
    internal const string ParserMismatchDelimiter = "ParserMismatchDelimiter";
    internal const string ParserDanglingClause = "ParserDanglingClause";
    internal const string UnknownDefinitionTag = "UnknownDefinitionTag";
    internal const string CDATASection = "CDATASection";
    internal const string TextSection = "TextSection";
    internal const string IncorrectSyntax = "IncorrectSyntax";
    internal const string IncorrectTypeSyntax = "IncorrectTypeSyntax";
    internal const string Error_MultipleRootActivityCreator = "Error_MultipleRootActivityCreator";
    internal const string Error_MustHaveParent = "Error_MustHaveParent";

    // Workflow References
    internal const string Error_ReferenceObjNotInitialized = "Error_ReferenceObjNotInitialized";
    internal const string Error_ReferenceInitResourceManager = "Error_ReferenceInitResourceManager";
    internal const string Error_ResourceReferenceGetObject = "Error_ResourceReferenceGetObject";
    internal const string Error_RefBindCantFindRef = "Error_RefBindCantFindRef";
    internal const string Error_RefBindMissingReferenceName = "Error_RefBindMissingReferenceName";
    internal const string Error_RefBindMissingAttribute = "Error_RefBindMissingAttribute";
    internal const string Error_ReferenceLoad = "Error_ReferenceLoad";
    internal const string Error_ReferenceMissingAttribute = "Error_ReferenceMissingAttribute";
    internal const string Error_ReferenceInvalidResourceFile = "Error_ReferenceInvalidResourceFile";
    internal const string Error_ReferenceEmptyName = "Error_ReferenceEmptyName";

    internal const string HandlerInvokerName = "HandlerInvokerName";
    internal const string HandlerInvokerSwitchPrefix_CS = "HandlerInvokerSwitchPrefix_CS";
    internal const string HandlerInvokerSwitchPrefix_VB = "HandlerInvokerSwitchPrefix_VB";
    internal const string HandlerInvokerSwitchSuffix_CS = "HandlerInvokerSwitchSuffix_CS";
    internal const string HandlerInvokerSwitchSuffix_VB = "HandlerInvokerSwitchSuffix_VB";
    internal const string HandlerInvokerCaseBegin_CS = "HandlerInvokerCaseBegin_CS";
    internal const string HandlerInvokerCaseBegin_VB = "HandlerInvokerCaseBegin_VB";

    // Activity Category
    internal const string Standard = "Standard";
    internal const string Base = "Base";

    //CustomActivityDesigner
    internal const string ValidatorCompanionClassDesc = "ValidatorCompanionClassDesc";
    internal const string ExecutorCompanionClassDesc = "ExecutorCompanionClassDesc";
    internal const string DesignerCompanionClassDesc = "DesignerCompanionClassDesc";
    internal const string CustomActivityBaseTypeDesc = "CustomActivityBaseTypeDesc";
    internal const string ActivityProperties = "ActivityProperties";
    internal const string ActivityPropertiesDesc = "ActivityPropertiesDesc";
    internal const string CompanionClasses = "CompanionClasses";
    internal const string ActivityDesc = "Activity";
    internal const string Error_TypeConversionFailed = "Error_TypeConversionFailed";
    internal const string SupportDataContext = "SupportDataContext";
    internal const string AdvancedCategory = "AdvancedCategory";
    internal const string SupportDataContextDesc = "SupportDataContextDesc";
    internal const string BaseCompanionClassName = "BaseCompanionClassName";
    internal const string BaseCompanionClassDesc = "BaseCompanionClassDesc";
    internal const string Designer = "Designer";
    internal const string Validator = "Validator";
    internal const string Executor = "Executor";
    internal const string BaseActivityType = "BaseActivityType";
    internal const string Error_NotBuiltInActivity = "Error_NotBuiltInActivity";
    internal const string NoChildActivities_Message = "NoChildActivities_Message";
    internal const string NoChildActivities_Caption = "NoChildActivities_Caption";
    internal const string Error_CustomActivityCantCreate = "Error_CustomActivityCantCreate";
    internal const string Error_CantChangeBuiltInActivity = "Error_CantChangeBuiltInActivity";
    internal const string Error_CantAddBeforeBuiltInActivity = "Error_CantAddBeforeBuiltInActivity";
    internal const string Error_CantAddAfterNonBuiltInActivity = "Error_CantAddAfterNonBuiltInActivity";
    internal const string Error_CannotAddRemoveChildActivities = "Error_CannotAddRemoveChildActivities";
    internal const string Error_CantFindBuiltInActivity = "Error_CantFindBuiltInActivity";
    internal const string Error_MissingBaseCompanionClassAttribute = "Error_MissingBaseCompanionClassAttribute";
    internal const string Error_CantFindBuiltInParent = "Error_CantFindBuiltInParent";
    internal const string Error_CantCreateInstanceOfBaseType = "Error_CantCreateInstanceOfBaseType";
    internal const string Error_CustomActivityTypeCouldNotBeFound = "Error_CustomActivityTypeCouldNotBeFound";
    internal const string None = "None";
    internal const string AtomicTransaction = "AtomicTransaction";
    internal const string LocalDataContext = "LocalDataContext";
    internal const string LocalDataContextDesc = "LocalDataContextDesc";
    internal const string CompanionClass = "CompanionClass";
    internal const string Error_AlreadyRootActivity = "Error_AlreadyRootActivity";
    internal const string RootActivityName = "RootActivityName";
    internal const string RootActivityNameDesc = "RootActivityNameDesc";
    internal const string CustomProperties = "CustomProperties";
    internal const string VisibleDescr = "VisibleDescr";
    internal const string EditableDescr = "EditableDescr";
    internal const string Error_CantCreateMethod = "Error_CantCreateMethod";
    internal const string Error_CantEditNullValue = "Error_CantEditNullValue";
    internal const string Error_CompanionTypeNotSet = "Error_CompanionTypeNotSet";
    internal const string Error_CompanionClassNameCanNotBeEmpty = "Error_CompanionClassNameCanNotBeEmpty";
    internal const string Error_CouldNotEmitFieldInLocalDataContext = "Error_CouldNotEmitFieldInLocalDataContext";
    internal const string Error_CouldNotEmitMethodInLocalDataContext = "Error_CouldNotEmitMethodInLocalDataContext";
    internal const string Error_DerivationFromTypeWithLocalDataContext = "Error_DerivationFromTypeWithLocalDataContext";
    internal const string Error_CompanionTypeDerivationError = "Error_CompanionTypeDerivationError";
    internal const string Error_CantCreateDataContextClass = "Error_CantCreateDataContextClass";
    internal const string ArrayExistingBind = "ArrayExistingBind";
    internal const string Error_NoMatchingFieldsOrProperties = "Error_NoMatchingFieldsOrProperties";
    internal const string ChooseFieldPropertyDatasource = "ChooseFieldPropertyDatasource";

    internal const string SupportsTransaction = "SupportsTransaction";
    internal const string SupportsExceptions = "SupportsExceptions";
    internal const string SupportsCancellationHandlerActivity = "SupportsCancellationHandlerActivity";
    internal const string SupportsEvents = "SupportsEvents";
    internal const string SupportsDataSources = "SupportsDataSources";
    internal const string SupportsCompensationHandler = "SupportsCompensationHandler";
    internal const string SupportsCompensationHandlerDesc = "SupportsCompensationHandlerDesc";
    internal const string SupportsTransactionDesc = "SupportsTransactionDesc";
    internal const string SupportsExceptionsDesc = "SupportsExceptionsDesc";
    internal const string SupportsCancelHandlerDesc = "SupportsCancelHandlerDesc";
    internal const string SupportsEventsDesc = "SupportsEventsDesc";
    internal const string TransactionDesc = "TransactionDesc";

    internal const string Error_BaseTypeMustBeActivity = "Error_BaseTypeMustBeActivity";
    internal const string ExistingActivityBindTitle = "ExistingActivityBindTitle";
    internal const string ExistingActivityBindLabel = "ExistingActivityBindLabel";
    internal const string ExistingFieldPropBindTitle = "ExistingFieldPropBindTitle";
    internal const string ExistingFieldPropBindLabel = "ExistingFieldPropBindLabel";
    internal const string ProvidesSynchronization = "ProvidesSynchronization";
    internal const string ProvidesSynchronizationDesc = "ProvidesSynchronizationDesc";
    internal const string SynchronizationHandles = "SynchronizationHandles";
    internal const string SynchronizationHandlesDesc = "SynchronizationHandlesDesc";

    internal const string Error_TransactionAlreadyApplied = "Error_TransactionAlreadyApplied";
    internal const string Error_BindBaseTypeNotSpecified = "Error_BindBaseTypeNotSpecified";
    internal const string NonDelegateTargetType = "NonDelegateTargetType";
    internal const string Error_ClassnameNotInRootNamespace = "Error_ClassnameNotInRootNamespace";
    internal const string Error_CantUseCurrentProjectTypeAsBase = "Error_CantUseCurrentProjectTypeAsBase";
    internal const string Error_UnboundGenericType = "Error_UnboundGenericType";
    internal const string Error_UnboundGenericTypeDataSource = "Error_UnboundGenericTypeDataSource";
    internal const string Error_BaseTypeUnknown = "Error_BaseTypeUnknown";
    internal const string Error_UnconfiguredBind = "Error_UnconfiguredBind";
    internal const string Error_CanNotEmitMemberInLocalDataContext = "Error_CanNotEmitMemberInLocalDataContext";
    internal const string Error_DesignedTypeNotFound = "Error_DesignedTypeNotFound";
    internal const string Error_PathCouldNotBeResolvedToMember = "Error_PathCouldNotBeResolvedToMember";
    internal const string Error_EdittingNullCollection = "Error_EdittingNullCollection";
    internal const string Error_MoreThanOneCompensationDecl = "Error_MoreThanOneCompensationDecl";
    internal const string Error_ParentDoesNotSupportCompensation = "Error_ParentDoesNotSupportCompensation";
    internal const string Error_CantResolveEventHandler = "Error_CantResolveEventHandler";
    internal const string Error_XSDObjectTypeNotSerializable = "Error_XSDObjectTypeNotSerializable";
    internal const string AEC_InvalidActivity = "AEC_InvalidActivity";
    internal const string GetDynamicActivities_InvalidActivity = "GetDynamicActivities_InvalidActivity";
    internal const string AEC_InvalidNestedActivity = "AEC_InvalidNestedActivity";
    internal const string Error_IDNotSetForActivitySource = "Error_IDNotSetForActivitySource";
    internal const string Error_InvalidCustomPropertyName = "Error_InvalidCustomPropertyName";
    internal const string Error_InvalidCustomPropertyType = "Error_InvalidCustomPropertyType";

    internal const string Error_DPReadOnly = "Error_DPReadOnly";
    internal const string Error_DPMetaPropertyBinding = "Error_DPMetaPropertyBinding";
    internal const string Error_DPSetValueBind = "Error_DPSetValueBind";
    internal const string Error_DPSetValueHandler = "Error_DPSetValueHandler";
    internal const string Error_DPGetValueHandler = "Error_DPGetValueHandler";
    internal const string Error_DPAddHandlerNonEvent = "Error_DPAddHandlerNonEvent";
    internal const string Error_DPAddHandlerMetaProperty = "Error_DPAddHandlerMetaProperty";
    internal const string Error_DPRemoveHandlerBind = "Error_DPRemoveHandlerBind";
    internal const string Error_LanguageNeedsToBeVBCSharp = "Error_LanguageNeedsToBeVBCSharp";
    internal const string Error_TargetFxNotSupported = "Error_TargetFxNotSupported";
    internal const string Error_CantConvertValueValue = "Error_CantConvertValueValue";
    internal const string Error_TypeIsNotValid = "Error_TypeIsNotValid";
    internal const string Error_TypePropertyInvalid = "Error_TypePropertyInvalid";
    internal const string Error_EventCantBeMetaProperty = "Error_EventCantBeMetaProperty";
    internal const string Error_EventMustBeDelegate = "Error_EventMustBeDelegate";
    internal const string Error_DPPropertyTypeMissing = "Error_DPPropertyTypeMissing";

    internal const string TransactionalContextActivityDescription = "TransactionalContextActivityDescription";
    internal const string CompensatableTransactionalContextActivityDescription = "CompensatableTransactionalContextActivityDescription";
    internal const string SynchronizationScopeActivityDescription = "SynchronizationScopeActivityDescription";
    internal const string SequenceActivityDescription = "SequenceActivityDescription";
    internal const string CompensateActivityDescription = "CompensateActivityDescription";
    internal const string Error_CompensateBadTargetTX = "Error_CompensateBadTargetTX";
    internal const string Error_CancelHandlerParentNotScope = "Error_CancelHandlerParentNotScope";
    internal const string FaultHandlerActivityDescription = "FaultHandlerActivityDescription";
    internal const string Error_ExceptionTypeNotException = "Error_ExceptionTypeNotException";
    internal const string Error_FaultIsNotOfFaultType = "Error_FaultIsNotOfFaultType";
    internal const string Error_FaultTypeNoDefaultConstructor = "Error_FaultTypeNoDefaultConstructor";
    internal const string FilterDescription_FaultHandlerActivity = "FilterDescription_FaultHandlerActivity";
    internal const string Error_FaultHandlerActivityParentNotFaultHandlersActivity = "Error_FaultHandlerActivityParentNotFaultHandlersActivity";
    internal const string Error_FaultHandlerActivityAllMustBeLast = "Error_FaultHandlerActivityAllMustBeLast";
    internal const string Error_FaultHandlersActivityDeclNotAllFaultHandlerActivityDecl = "Error_FaultHandlersActivityDeclNotAllFaultHandlerActivityDecl";
    internal const string Error_FaultHandlerActivityWrongOrder = "Error_FaultHandlerActivityWrongOrder";
    internal const string Error_SenderMustBeActivityExecutionContext = "Error_SenderMustBeActivityExecutionContext";
    internal const string Error_XomlWorkflowHasCode = "Error_XomlWorkflowHasCode";
    internal const string Error_WrongParamForActivityResolveEventArgs = "Error_WrongParamForActivityResolveEventArgs";
    internal const string Error_ValidatorThrewException = "Error_ValidatorThrewException";
    internal const string Error_Missing_CanModifyProperties_True = "Error_Missing_CanModifyProperties_True";
    internal const string Error_Missing_CanModifyProperties_False = "Error_Missing_CanModifyProperties_False";
    internal const string Error_ModelingConstructsCanNotContainModelingConstructs = "Error_ModelingConstructsCanNotContainModelingConstructs";
    internal const string Error_RootIsNotEnabled = "Error_RootIsNotEnabled";
    internal const string Error_MissingSetAccessor = "Error_MissingSetAccessor";
    internal const string Error_MissingAddHandler = "Error_MissingAddHandler";
    internal const string Error_MissingCLRProperty = "Error_MissingCLRProperty";

    internal const string Error_NotReadOnlyProperty = "Error_NotReadOnlyProperty";
    internal const string Error_InvalidDependencyProperty = "Error_InvalidDependencyProperty";
    internal const string Error_ActivityNameExist = "Error_ActivityNameExist";
    internal const string CannotCreateAttribute = "CannotCreateAttribute";
    internal const string NamespaceAndDeclaringTypeCannotBeNull = "NamespaceAndDeclaringTypeCannotBeNull";
    internal const string NotElementType = "NotElementType";

    //Layout persistence errors
    internal const string Error_LayoutSerializationActivityNotFound = "Error_LayoutSerializationActivityNotFound";
    internal const string Error_LayoutSerializationAssociatedActivityNotFound = "Error_LayoutSerializationAssociatedActivityNotFound";
    internal const string Error_LayoutSerializationPersistenceSupport = "Error_LayoutSerializationPersistenceSupport";
    internal const string Error_LayoutSerializationRootDesignerNotFound = "Error_LayoutSerializationRootDesignerNotFound";
    internal const string Error_ParameterCannotBeEmpty = "Error_ParameterCannotBeEmpty";
    internal const string InvalidExecutionStatus = "InvalidExecutionStatus";
    internal const string Error_LayoutDeserialization = "Error_LayoutDeserialization";
    internal const string Error_LayoutSerialization = "Error_LayoutSerialization";

    internal const string Error_SerializerStackOverflow = "Error_SerializerStackOverflow";
    internal const string Error_InvalidActivityForWorkflowChanges = "Error_InvalidActivityForWorkflowChanges";
    internal const string Error_InvalidMemberType = "Error_InvalidMemberType";
    internal const string Error_BindPathNullValue = "Error_BindPathNullValue";
    internal const string Error_MarkupExtensionMissingTerminatingCharacter = "Error_MarkupExtensionMissingTerminatingCharacter";
    internal const string Error_MarkupExtensionDeserializeFailed = "Error_MarkupExtensionDeserializeFailed";
    internal const string Error_ApplyDynamicChangeFailed = "Error_ApplyDynamicChangeFailed";
    internal const string Error_ActivityCircularReference = "Error_ActivityCircularReference";
    internal const string Error_ValidatorTypeIsInvalid = "Error_ValidatorTypeIsInvalid";
    internal const string Error_InvalidServiceProvider = "Error_InvalidServiceProvider";
    internal const string Error_InvalidRootForWorkflowChanges = "Error_InvalidRootForWorkflowChanges";
    internal const string Error_ExtraCharacterFoundAtEnd = "Error_ExtraCharacterFoundAtEnd";
    internal const string Error_WorkflowChangesNotSupported = "Error_WorkflowChangesNotSupported";
    internal const string Error_TypeSystemAttributeArgument = "Error_TypeSystemAttributeArgument";

    internal const string Error_InvalidElementFoundForType = "Error_InvalidElementFoundForType";
    internal const string Error_ActivitySaveLoadNotCalled = "Error_ActivitySaveLoadNotCalled";
    internal const string Error_CanNotBindProperty = "Error_CanNotBindProperty";
}
