// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Diagnostics.CodeAnalysis;
using JetBrains.Annotations;

namespace Microsoft.EntityFrameworkCore.Query;

/// <summary>
///     <para>
///         A factory to create transparent identifier to create during query processing.
///         Transparent identifier is struct of outer and inner elements which is generally created as a result of join methods
///         as intermediate type to hold values from both sources.
///     </para>
///     <para>
///         This type is typically used by database providers (and other extensions). It is generally
///         not used in application code.
///     </para>
/// </summary>
/// <remarks>
///     See <see href="https://aka.ms/efcore-docs-providers">Implementation of database providers and extensions</see>
///     and <see href="https://aka.ms/efcore-docs-how-query-works">How EF Core queries work</see> for more information and examples.
/// </remarks>
public static class TransparentIdentifierFactory
{
    /// <summary>
    ///     Creates new transparent identifier type for given types.
    /// </summary>
    /// <param name="outerType">The outer type of the transparent identifier.</param>
    /// <param name="innerType">The inner type of the transparent identifier.</param>
    /// <returns>The created transparent identifier type.</returns>
    [return: DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)]
    [DynamicDependency(DynamicallyAccessedMemberTypes.All, typeof(TransparentIdentifier<,>))]
    public static Type Create(Type outerType, Type innerType)
        => typeof(TransparentIdentifier<,>).MakeGenericType(outerType, innerType);

    private readonly struct TransparentIdentifier<TOuter, TInner>(TOuter outer, TInner inner)
    {
        /// <summary>
        ///     This is an internal API that supports the Entity Framework Core infrastructure and not subject to
        ///     the same compatibility standards as public APIs. It may be changed or removed without notice in
        ///     any release. You should only use it directly in your code with extreme caution and knowing that
        ///     doing so can result in application failures when updating to a new Entity Framework Core release.
        /// </summary>
        [UsedImplicitly]
        public readonly TOuter Outer = outer;

        /// <summary>
        ///     This is an internal API that supports the Entity Framework Core infrastructure and not subject to
        ///     the same compatibility standards as public APIs. It may be changed or removed without notice in
        ///     any release. You should only use it directly in your code with extreme caution and knowing that
        ///     doing so can result in application failures when updating to a new Entity Framework Core release.
        /// </summary>
        [UsedImplicitly]
        public readonly TInner Inner = inner;
    }
}
