// SPDX-License-Identifier: GPL-2.0 OR BSD-3-Clause
/* Copyright(c) 2024  Realtek Corporation
 */

#include "util.h"

#define RTW89_DBM_QUARTER_FACTOR 2
#define RTW89_MIN_DBM (-41.25 * (1 << RTW89_DBM_QUARTER_FACTOR))
#define RTW89_MAX_DBM (96 * (1 << RTW89_DBM_QUARTER_FACTOR))
#define RTW89_DB_INVERT_TABLE_OFFSET (-RTW89_MIN_DBM)

static const u64 db_invert_table[] = {
	/* in unit of 0.000001 */
	75, 79, 84, 89, 94, 100, 106, 112, 119, 126, 133, 141, 150, 158, 168, 178, 188,
	200, 211, 224, 237, 251, 266, 282, 299, 316, 335, 355, 376, 398, 422, 447, 473,
	501, 531, 562, 596, 631, 668, 708, 750, 794, 841, 891, 944, 1000, 1059, 1122, 1189,
	1259, 1334, 1413, 1496, 1585, 1679, 1778, 1884, 1995, 2113, 2239, 2371, 2512, 2661,
	2818, 2985, 3162, 3350, 3548, 3758, 3981, 4217, 4467, 4732, 5012, 5309, 5623, 5957,
	6310, 6683, 7079, 7499, 7943, 8414, 8913, 9441, 10000, 10593, 11220, 11885, 12589,
	13335, 14125, 14962, 15849, 16788, 17783, 18836, 19953, 21135, 22387, 23714, 25119,
	26607, 28184, 29854, 31623, 33497, 35481, 37584, 39811, 42170, 44668, 47315, 50119,
	53088, 56234, 59566, 63096, 66834, 70795, 74989, 79433, 84140, 89125, 94406, 100000,
	105925, 112202, 118850, 125893, 133352, 141254, 149624, 158489, 167880, 177828,
	188365, 199526, 211349, 223872, 237137, 251189, 266073, 281838, 298538, 316228,
	334965, 354813, 375837, 398107, 421697, 446684, 473151, 501187, 530884, 562341,
	595662, 630957, 668344, 707946, 749894, 794328, 841395, 891251, 944061, 1000000,
	1059254, 1122018, 1188502, 1258925, 1333521, 1412538, 1496236, 1584893, 1678804,
	1778279, 1883649, 1995262, 2113489, 2238721, 2371374, 2511886, 2660725, 2818383,
	2985383, 3162278, 3349654, 3548134, 3758374, 3981072, 4216965, 4466836, 4731513,
	5011872, 5308844, 5623413, 5956621, 6309573, 6683439, 7079458, 7498942, 7943282,
	8413951, 8912509, 9440609, 10000000, 10592537, 11220185, 11885022, 12589254,
	13335214, 14125375, 14962357, 15848932, 16788040, 17782794, 18836491, 19952623,
	21134890, 22387211, 23713737, 25118864, 26607251, 28183829, 29853826, 31622777,
	33496544, 35481339, 37583740, 39810717, 42169650, 44668359, 47315126, 50118723,
	53088444, 56234133, 59566214, 63095734, 66834392, 70794578, 74989421, 79432823,
	84139514, 89125094, 94406088, 100000000, 105925373, 112201845, 118850223, 125892541,
	133352143, 141253754, 149623566, 158489319, 167880402, 177827941, 188364909, 199526231,
	211348904, 223872114, 237137371, 251188643, 266072506, 281838293, 298538262, 316227766,
	334965439, 354813389, 375837404, 398107171, 421696503, 446683592, 473151259, 501187234,
	530884444, 562341325, 595662144, 630957344, 668343918, 707945784, 749894209, 794328235,
	841395142, 891250938, 944060876, 1000000000, 1059253725, 1122018454, 1188502227,
	1258925412, 1333521432, 1412537545, 1496235656, 1584893192, 1678804018, 1778279410,
	1883649089, 1995262315, 2113489040, 2238721139, 2371373706, 2511886432, 2660725060,
	2818382931, 2985382619, 3162277660, 3349654392, 3548133892, 3758374043, 3981071706,
	4216965034, 4466835922ULL, 4731512590ULL, 5011872336ULL, 5308844442ULL, 5623413252ULL,
	5956621435ULL, 6309573445ULL, 6683439176ULL, 7079457844ULL, 7498942093ULL,
	7943282347ULL, 8413951416ULL, 8912509381ULL, 9440608763ULL, 10000000000ULL,
	10592537252ULL, 11220184543ULL, 11885022274ULL, 12589254118ULL, 13335214322ULL,
	14125375446ULL, 14962356561ULL, 15848931925ULL, 16788040181ULL, 17782794100ULL,
	18836490895ULL, 19952623150ULL, 21134890398ULL, 22387211386ULL, 23713737057ULL,
	25118864315ULL, 26607250598ULL, 28183829313ULL, 29853826189ULL, 31622776602ULL,
	33496543916ULL, 35481338923ULL, 37583740429ULL, 39810717055ULL, 42169650343ULL,
	44668359215ULL, 47315125896ULL, 50118723363ULL, 53088444423ULL, 56234132519ULL,
	59566214353ULL, 63095734448ULL, 66834391757ULL, 70794578438ULL, 74989420933ULL,
	79432823472ULL, 84139514165ULL, 89125093813ULL, 94406087629ULL, 100000000000ULL,
	105925372518ULL, 112201845430ULL, 118850222744ULL, 125892541179ULL, 133352143216ULL,
	141253754462ULL, 149623565609ULL, 158489319246ULL, 167880401812ULL, 177827941004ULL,
	188364908949ULL, 199526231497ULL, 211348903984ULL, 223872113857ULL, 237137370566ULL,
	251188643151ULL, 266072505980ULL, 281838293126ULL, 298538261892ULL, 316227766017ULL,
	334965439158ULL, 354813389234ULL, 375837404288ULL, 398107170553ULL, 421696503429ULL,
	446683592151ULL, 473151258961ULL, 501187233627ULL, 530884444231ULL, 562341325190ULL,
	595662143529ULL, 630957344480ULL, 668343917569ULL, 707945784384ULL, 749894209332ULL,
	794328234724ULL, 841395141645ULL, 891250938134ULL, 944060876286ULL, 1000000000000ULL,
	1059253725177ULL, 1122018454302ULL, 1188502227437ULL, 1258925411794ULL,
	1333521432163ULL, 1412537544623ULL, 1496235656094ULL, 1584893192461ULL,
	1678804018123ULL, 1778279410039ULL, 1883649089490ULL, 1995262314969ULL,
	2113489039837ULL, 2238721138568ULL, 2371373705662ULL, 2511886431510ULL,
	2660725059799ULL, 2818382931264ULL, 2985382618918ULL, 3162277660168ULL,
	3349654391578ULL, 3548133892336ULL, 3758374042884ULL, 3981071705535ULL,
	4216965034286ULL, 4466835921510ULL, 4731512589615ULL, 5011872336273ULL,
	5308844442310ULL, 5623413251904ULL, 5956621435290ULL, 6309573444802ULL,
	6683439175686ULL, 7079457843841ULL, 7498942093325ULL, 7943282347243ULL,
	8413951416452ULL, 8912509381337ULL, 9440608762859ULL, 10000000000000ULL,
	10592537251773ULL, 11220184543020ULL, 11885022274370ULL, 12589254117942ULL,
	13335214321633ULL, 14125375446228ULL, 14962356560944ULL, 15848931924611ULL,
	16788040181226ULL, 17782794100389ULL, 18836490894898ULL, 19952623149689ULL,
	21134890398367ULL, 22387211385683ULL, 23713737056617ULL, 25118864315096ULL,
	26607250597988ULL, 28183829312645ULL, 29853826189180ULL, 31622776601684ULL,
	33496543915783ULL, 35481338923358ULL, 37583740428845ULL, 39810717055350ULL,
	42169650342858ULL, 44668359215096ULL, 47315125896148ULL, 50118723362727ULL,
	53088444423099ULL, 56234132519035ULL, 59566214352901ULL, 63095734448019ULL,
	66834391756862ULL, 70794578438414ULL, 74989420933246ULL, 79432823472428ULL,
	84139514164520ULL, 89125093813375ULL, 94406087628593ULL, 100000000000000ULL,
	105925372517729ULL, 112201845430197ULL, 118850222743702ULL, 125892541179417ULL,
	133352143216332ULL, 141253754462276ULL, 149623565609444ULL, 158489319246111ULL,
	167880401812256ULL, 177827941003893ULL, 188364908948981ULL, 199526231496888ULL,
	211348903983664ULL, 223872113856834ULL, 237137370566166ULL, 251188643150958ULL,
	266072505979882ULL, 281838293126446ULL, 298538261891796ULL, 316227766016838ULL,
	334965439157829ULL, 354813389233577ULL, 375837404288444ULL, 398107170553497ULL,
	421696503428583ULL, 446683592150964ULL, 473151258961482ULL, 501187233627272ULL,
	530884444230989ULL, 562341325190350ULL, 595662143529011ULL, 630957344480196ULL,
	668343917568615ULL, 707945784384138ULL, 749894209332456ULL, 794328234724284ULL,
	841395141645198ULL, 891250938133745ULL, 944060876285923ULL, 1000000000000000ULL,
	1059253725177290ULL, 1122018454301970ULL, 1188502227437020ULL, 1258925411794170ULL,
	1333521432163330ULL, 1412537544622760ULL, 1496235656094440ULL, 1584893192461110ULL,
	1678804018122560ULL, 1778279410038920ULL, 1883649089489810ULL, 1995262314968890ULL,
	2113489039836650ULL, 2238721138568340ULL, 2371373705661660ULL, 2511886431509590ULL,
	2660725059798820ULL, 2818382931264460ULL, 2985382618917960ULL, 3162277660168380ULL,
	3349654391578280ULL, 3548133892335770ULL, 3758374042884440ULL, 3981071705534970ULL
};

s32 rtw89_linear_to_db_quarter(u64 val)
{
	int r = ARRAY_SIZE(db_invert_table) - 1;
	int l = 0;
	int m;

	while (l <= r) {
		m = l + (r - l) / 2;

		if (db_invert_table[m] == val)
			return m - (s32)RTW89_DB_INVERT_TABLE_OFFSET;

		if (db_invert_table[m] > val)
			r = m - 1;
		else
			l = m + 1;
	}

	if (l >= ARRAY_SIZE(db_invert_table))
		return RTW89_MAX_DBM;
	else if (r < 0)
		return RTW89_MIN_DBM;
	else if (val - db_invert_table[r] <= db_invert_table[l] - val)
		return r - (s32)RTW89_DB_INVERT_TABLE_OFFSET;
	else
		return l - (s32)RTW89_DB_INVERT_TABLE_OFFSET;
}
EXPORT_SYMBOL(rtw89_linear_to_db_quarter);

s32 rtw89_linear_to_db(u64 val)
{
	return rtw89_linear_to_db_quarter(val) >> RTW89_DBM_QUARTER_FACTOR;
}
EXPORT_SYMBOL(rtw89_linear_to_db);

u64 rtw89_db_quarter_to_linear(s32 db)
{
	/* supported range -41.25 to 96 dBm, in unit of 0.25 dBm */
	db = clamp_t(s32, db, RTW89_MIN_DBM, RTW89_MAX_DBM);
	db += (s32)RTW89_DB_INVERT_TABLE_OFFSET;

	return db_invert_table[db];
}
EXPORT_SYMBOL(rtw89_db_quarter_to_linear);

u64 rtw89_db_to_linear(s32 db)
{
	return rtw89_db_quarter_to_linear(db << RTW89_DBM_QUARTER_FACTOR);
}
EXPORT_SYMBOL(rtw89_db_to_linear);

void rtw89_might_trailing_ellipsis(char *buf, size_t size, ssize_t used)
{
	static const char ellipsis[] = "...";

	/* length of null terminiator isn't included in 'used' */
	if (used + 1 < size || size < sizeof(ellipsis))
		return;

	memcpy(buf + size - sizeof(ellipsis), ellipsis, sizeof(ellipsis));
}
